/************************************************************************************************************\

Module Name:    LNetStreamer.h

Description:    Liberatus Network Streamer API

References:     LNetStreamer.doc Revision 0.1

    Copyright (c) 2014, Matrox Graphics Inc.
    All Rights Reserved.

\************************************************************************************************************/


#ifndef INC_LNETSTREAMER_H
#define INC_LNETSTREAMER_H


/* -------------------------------------------------------------------------------------------------------- */
/*                                   I N C L U D E S   A N D   U S I N G S */
/* -------------------------------------------------------------------------------------------------------- */


#include "Liberatus.h"


/* -------------------------------------------------------------------------------------------------------- */
/*                                   C O N S T A N T S   A N D   T Y P E S */
/* -------------------------------------------------------------------------------------------------------- */


#define LNETSTREAMER_MEDIA_MAX_INTERNAL_DATA            6u
#define LNETSTREAMER_VIDEO_BUFFER_COUNT_DEFAULT         0u
#define LNETSTREAMER_VIDEO_BUFFER_LENGTH_DEFAULT        0u
#define LNETSTREAMER_AUDIO_BUFFER_COUNT_DEFAULT         0u
#define LNETSTREAMER_AUDIO_BUFFER_LENGTH_DEFAULT        0u
#define LNETSTREAMER_METADATA_BUFFER_COUNT_DEFAULT      0u
#define LNETSTREAMER_METADATA_BUFFER_LENGTH_DEFAULT     0u
#define LNETSTREAMER_MAX_NUMBER_OF_AUDIO_TRACK          16u
#define LNETSTREAMER_MAX_STREAMID_LENGTH                512u


/************************************************************************************************************\

Structure:      LNetStreamer_Handle

Description:    Network stream handle

Comments:       Members are private to LNetStreamer implementation.

\************************************************************************************************************/
        struct tagLNetStreamer_Private;
typedef struct tagLNetStreamer_Private* LNetStreamer_Handle;


/************************************************************************************************************\

Enum:           LNetStreamer_Service

Description:    Service provided by network stream.

Comments:

\************************************************************************************************************/
typedef enum
{
    LNetStreamer_Service_CLIENT                 = 1,        /* Network stream is a client */
    LNetStreamer_Service_SERVER                 = 2         /* Network stream is a server */
} LNetStreamer_Service;


/************************************************************************************************************\

Enum:           LNetStreamer_Protocol

Description:    Protocol used by network stream.

Comments:

\************************************************************************************************************/
typedef enum
{
    LNetStreamer_Protocol_RTSP                  = 1,        /* Network stream uses RTSP protocol */
    LNetStreamer_Protocol_RTP                   = 2,        /* Network stream uses RTP protocol */
    LNetStreamer_Protocol_TS                    = 3,        /* Network stream uses TS protocol */
    LNetStreamer_Protocol_RTMP                  = 4,        /* Network stream uses RTMP protocol */
    LNetStreamer_Protocol_HLS                   = 5         /* Network stream uses HLS protocol (server only) */
} LNetStreamer_Protocol;


/************************************************************************************************************\

Enum:           LNetStreamer_SrtMode

Description:    SRT connection modes.

Comments:

\************************************************************************************************************/
typedef enum
{
    LNetStreamer_SrtMode_LISTENER               = 1,        /* Wait for connection */
    LNetStreamer_SrtMode_CALLER                 = 2,        /* Initiate connection */
    LNetStreamer_SrtMode_RENDEZVOUS             = 3         /* Initiate and wait for connection */
} LNetStreamer_SrtMode;


/************************************************************************************************************\

Enum:           LNetStreamer_MediaType

Description:    Type of media information structure.

Comments:

\************************************************************************************************************/
typedef enum
{
    LNetStreamer_MediaType_VIDEO                = 1,        /* Media information specifying video data */
    LNetStreamer_MediaType_AUDIO                = 2,        /* Media information specifying audio data */
    LNetStreamer_MediaType_METADATA             = 3         /* Media information specifying meta data */
} LNetStreamer_MediaType;


/************************************************************************************************************\

Structure:      LNetStreamer_Media

Description:    Media information structure

Comments:

\************************************************************************************************************/
typedef struct tagLNetStreamer_Media
{
    LNetStreamer_MediaType    eMediaType;       /* Type of media information structure.  Set the member to */
                                                /* LNetStreamer_MediaType_VIDEO, LNetStreamer_MediaType_AUDIO or */
                                                /* LNetStreamer_MediaType_METADATA. */
    LBuffer_Handle          hBuffer;            /* Handle of LBuffer.  For client session, read data from this */
                                                /* handle.  For server sessions, write data into this handle. */
    MUINT8*                 paucData;           /* Optional pointer to buffer to be used when hBuffer member */
                                                /* is NULL. */
    MUINT32                 uiDataMaxLength;    /* Maximum length in bytes of available data in buffer. */
    MUINT32                 uiDataOffset;       /* Offset in bytes from start of LBuffer.  This member must be */
                                                /* ignored when hBuffer member is NULL. */
    MUINT32                 uiDataLength;       /* Length in bytes of valid data in buffer.  For client */
                                                /* session, this member is set on return from LNetStreamer_Get- */
                                                /* Media function.  For server session, this member must be */
                                                /* set by application before call to LNetStreamer_ReleaseMedia. */
    MUINT32                 uiPTS;              /* Presentation timestamp in milliseconds.  For client session, */
                                                /* this member is set by library on return from LNetStreamer_- */
                                                /* GetMedia function.  For server sessions, this member must */
                                                /* be set by application before call to LNetStreamer_Release- */
                                                /* Media. */
    MUINT32                 uiDTS;              /* Decode timestamp in milliseconds. */
    MUINT32                 uiTrackIndex;       /* Zero-based index of track */
    MUINT32                 auiInternalData[LNETSTREAMER_MEDIA_MAX_INTERNAL_DATA];
                                                /* Reserved for use by implementation. */
} LNetStreamer_Media;


/************************************************************************************************************\

Enum:           LNetStreamer_ConfigFlags

Description:    Select optional members of network stream configuration structure.

Comments:

\************************************************************************************************************/
typedef enum
{
    LNetStreamer_ConfigFlags_VIDEO_PORT         = 0x00000001,   /* Select uiVideoPort member. */
    LNetStreamer_ConfigFlags_AUDIO_PORT         = 0x00000002,   /* Select uiAudioPort member. */
    LNetStreamer_ConfigFlags_METADATA_PORT      = 0x00000004,   /* Select uiMetadataPort member. */
    LNetStreamer_ConfigFlags_LOCATION           = 0x00000008,   /* Select pacLocation and uiLocationLength */
                                                                /* members. */
    LNetStreamer_ConfigFlags_DESTINATION        = 0x00000010,   /* Select pacDestination and uiDestinationLength */
                                                                /* members. */
    LNetStreamer_ConfigFlags_VIDEO_HEADERS      = 0x00000020,   /* Select paucVideoHeaderPps, paucVideoHeaderPps, */
                                                                /* uiVideoHeaderPpsLength, uiVideoHeaderSpsLength */
                                                                /* members. */
    LNetStreamer_ConfigFlags_AUDIO_HEADER       = 0x00000040,   /* Select paucAudioHeader and uiAudioHeaderLength */
                                                                /* members. */
    LNetStreamer_ConfigFlags_SDP                = 0x00000080,   /* Select pacSdp uiSdpLength members. */
    LNetStreamer_ConfigFlags_NET_INTERFACE      = 0x00000100,   /* Select pacNetInterface and uiNetInterfaceLength */
                                                                /* members. */
    LNetStreamer_ConfigFlags_RTSP_PORT          = 0x00000200,   /* Select uiRtspPort member. */
    LNetStreamer_ConfigFlags_RTSP_FOLDER        = 0x00000400,   /* Select pacRtspFolder and uiRtspFolderLength */
                                                                /* members. */
    LNetStreamer_ConfigFlags_EVENT              = 0x00000800,   /* Select pfnEventCallback and pvEventContext */
                                                                /* members. */
    LNetStreamer_ConfigFlags_LBUFFER_AUTO_SYNC  = 0x00001000,   /* Enable auto-sync on LBuffer creation. */
    LNetStreamer_ConfigFlags_RTSP_SERVER        = 0x00002000,   /* Select bEnableRtspServer member */
    LNetStreamer_ConfigFlags_SPECIAL_1          = 0x00004000,   /* Magic trick #1 */
    LNetStreamer_ConfigFlags_RTMP_SERVER        = 0x00008000,   /* Select bEnableRtmpServer member */
    LNetStreamer_ConfigFlags_RTCP_FREQUENCY     = 0x00010000,   /* Select uiRtcpFrequency member */
    LNetStreamer_ConfigFlags_SRTP               = 0x00020000,   /* Select paucMasterKey, uiMasterKeyLength, */
                                                                /* paucMasterSalt, uiMasterSaltLength */
    LNetStreamer_ConfigFlags_NO_SKIP            = 0x00040000,   /* Disable skip of video frames */
    LNetStreamer_ConfigFlags_TTL                = 0x00080000,   /* Select uiTtl member */
    LNetStreamer_ConfigFlags_RTMP_APP           = 0x00100000,   /* Select pacRtmpApp and uiRtmpAppLength members. */
    LNetStreamer_ConfigFlags_MAX_BITRATE        = 0x00200000,   /* Select uiMaxBitrateKbps member. */
    LNetStreamer_ConfigFlags_RTSP_AUTH          = 0x00400000,   /* Select pacUsername, uiUsernameLength, */
                                                                /* pacPassword, uiPasswordLength members. */
    LNetStreamer_ConfigFlags_MTU                = 0x00800000,   /* Select uiMtu member */
    LNetStreamer_ConfigFlags_SERVER_VIDEO_PORT  = 0x01000000,   /* Select uiServerVideoPort member. */
    LNetStreamer_ConfigFlags_SERVER_AUDIO_PORT  = 0x02000000,   /* Select uiServerAudioPort member. */
    LNetStreamer_ConfigFlags_RAW                = 0x04000000,   /* Select raw data transfer mode. */
    LNetStreamer_ConfigFlags_SRT                = 0x08000000,   /* Enable SRT protocol */
    LNetStreamer_ConfigFlags_PRIVATE            = 0x10000000,   /* Enable private protocol */
    LNetStreamer_ConfigFlags_EXTRA_FLAGS        = 0x40000000    /* Select uiConfigFlags2 member. */
} LNetStreamer_ConfigFlags;


/************************************************************************************************************\

Enum:           LNetStreamer_ConfigFlags2

Description:    Select optional members of network stream configuration structure.

Comments:

\************************************************************************************************************/
typedef enum
{
    LNetStreamer_ConfigFlags2_SRT_LATENCY       = 0x00000001,   /* Select uiSrtLatencyMsec member. */
    LNetStreamer_ConfigFlags2_SRT_KEY_LENGTH    = 0x00000002,   /* Select uiSrtKeyLength member. */
    LNetStreamer_ConfigFlags2_NICE_INCREMENT    = 0x00000004,   /* Select iNiceIncrement member. */
    LNetStreamer_ConfigFlags2_MULTICAST_ONLY    = 0x00000008,   /* Support multicast as only transport for RTSP server. */
    LNetStreamer_ConfigFlags2_SSRC              = 0x00000010,   /* Select uiSSRC member. */
    LNetStreamer_ConfigFlags2_RTMP_TIMEOUT      = 0x00000020,   /* Select uiRtmpTimeout member. */
    LNetStreamer_ConfigFlags2_TCP_INTERLEAVED   = 0x00000040,   /* Enable TCP interleaved protocol */
    LNetStreamer_ConfigFlags2_HLS_FOLDER        = 0x00000080,   /* Select pacHlsFolder and uiHlsFolderLength members. */
    LNetStreamer_ConfigFlags2_HLS_DURATION      = 0x00000100,   /* Select uiHlsDuration member. */
    LNetStreamer_ConfigFlags2_HLS_COUNT         = 0x00000200,   /* Select uiHlsCount member. */
    LNetStreamer_ConfigFlags2_RTSP_WITH_HLS     = 0x00000400,   /* Enable HLS in combination with RTSP */
    LNetStreamer_ConfigFlags2_SSRC_AUTOMATIC    = 0x00000800,   /* Enable automatic detection and validation of SSRC. */
    LNetStreamer_ConfigFlags2_HLS_MASTER        = 0x00001000,   /* Select pacHlsMaster and uiHlsMasterLength members. */
    LNetStreamer_ConfigFlags2_CBR               = 0x00002000,   /* Enable CBR mode for TS protocol. */
    LNetStreamer_ConfigFlags2_IPV6              = 0x00004000,   /* Enable IPv6 sockets. */
    LNetStreamer_ConfigFlags2_SRT_MODE          = 0x00008000,   /* Select eSrtMode member. */
    LNetStreamer_ConfigFlags2_VIDEO_FRAME_RATE  = 0x00020000,   /* Select uiVideoFrameRateNum and uiVideoFrameRateDen. */
    LNetStreamer_ConfigFlags2_TRACK_COUNT       = 0x00040000,   /* Select uiVideoTrackCount and uiAudioTrackCount members. */
    LNetStreamer_ConfigFlags2_SOURCE            = 0x00100000,   /* Select pacSource and uiSourceLength members. */
    LNetStreamer_ConfigFlags2_EXTERNAL          = 0x00200000,   /* Use external library. */
    LNetStreamer_ConfigFlags2_RTP_TS            = 0x00800000,   /* Enable RTP layer for TS protocol */
    LNetStreamer_ConfigFlags2_TOS               = 0x02000000,   /* Select uiTos member. */
    LNetStreamer_ConfigFlags2_AUDIO_TRACK_PID   = 0x04000000,   /* Enable audio track PID insertion for TS/SRT protocol */
    LNetStreamer_ConfigFlags2_SRT_STREAMID      = 0x08000000,   /* Select pacStreamID, uiStreamIDLength */
    LNetStreamer_ConfigFlags2_EXTRA_FLAGS       = 0x40000000    /* Select uiConfigFlags3 member. */
} LNetStreamer_ConfigFlags2;


/************************************************************************************************************\

Enum:           LNetStreamer_EventType

Description:    Type of event

Comments:

\************************************************************************************************************/
typedef enum
{
    LNetStreamer_EventType_START                = 1,        /* Start sending media */
    LNetStreamer_EventType_STOP                 = 2,        /* Stop sending media */
    LNetStreamer_EventType_FRAME_SKIP           = 3,        /* Detected frame skip */
    LNetStreamer_EventType_MESSAGE              = 4,        /* Message to log */
    LNetStreamer_EventType_PACKET_LOSS          = 5,        /* Detected packet loss */
    LNetStreamer_EventType_SRT_STATISTICS       = 6,        /* Full SRT Statistics */
    LNetStreamer_EventType_RTSP_MESSAGE         = 7,        /* RTSP Message */
    LNetStreamer_EventType_REDO_CONNECTION      = 8,        /* Error Message, when connection is broken, use with MESSAGE to give more details */
    LNetStreamer_EventType_SRT_STREAMID         = 9,        /* SRT StreamID*/
} LNetStreamer_EventType;

typedef struct tagLNetStreamer_MessageEvent
{
    LNetStreamer_EventType                      eEventType;
    char                                        acMessageText[252u];
} LNetStreamer_MessageEvent;

typedef struct tagLNetStreamer_RtspMessageEvent
{
    LNetStreamer_EventType                      eEventType;
    MCHAR8*                                     pacMessageText;
    MUINT32                                     uiMessageLength;
} LNetStreamer_RtspMessageEvent;

/************************************************************************************************************\

Structure:      LNetStreamerSrtStatsAccumulated

Description:    Accumulated srt statistics structure.

\************************************************************************************************************/
typedef struct tagLNetStreamerSrtStatsAccumulated
{
    MUINT64 uiTimeStampMs;              /* The time elapsed, in milliseconds, since the SRT socket has been created. */
    MUINT64 uiTotalPackets;             /* The total number of sent/received DATA packets. */
    MUINT64 uiLossPackets;              /* The total number of data packets considered or reported as lost. */
    MUINT64 uiRetransPackets;           /* The total number of retransmitted packets sent by the SRT sender. */
                                        /* Available for sender. */
    MUINT64 uiAckPackets;               /* The total number of sent/received ACK (Acknowledgement) control packets. */
    MUINT64 uiNakPackets;               /* The total number of sent NAK (Negative Acknowledgement) control packets. */
    MUINT64 uiDropPackets;              /* The total number of dropped by the SRT sender/receiver DATA packets. */
    MUINT64 uiSndDurationUs;            /* The total accumulated time in microseconds, during which the SRT sender */
                                        /* has some data to transmit.*/
    MUINT64 uiRcvUndecryptPackets;      /* The total number of packets that failed to be decrypted at the receiver side.*/
    MUINT64 uiFilterExtraPackets;       /* The total number of packet filter control packets generated/received by */
                                        /* the packet filter.*/
    MUINT64 uiRcvFilterSupplyPackets;   /* The total number of lost DATA packets recovered by the packet filter at */
                                        /* the receiver side.*/
    MUINT64 uiRcvFilterLossPackets;     /* The total number of sent/received DATA packets for a specified interval.*/
} LNetStreamerSrtStatsAccumulated;

/************************************************************************************************************\

Structure:      LNetStreamerSrtStatsInterval

Description:    Interval srt statistics structure.

\************************************************************************************************************/
typedef struct tagLNetStreamerSrtStatsInterval
{
    MUINT64 uiTotalPackets;             /* The total number of data packets considered or reported as lost for a */
                                        /* specified interval. */
    MUINT64 uiLossPackets;              /* The total number of loss packets sent by the SRT sender for a specified interval. */
    MUINT64 uiRetransPackets;           /* The total number of retransmitted packets sent by the SRT sender for a specified */
                                        /* interval. */
    MUINT64 uiAckPackets;               /* The total number of sent/received ACK (Acknowledgement) control packets */
                                        /* for a specified interval. */
    MUINT64 uiNakPackets;               /* The total number of sent NAK (Negative Acknowledgement) control packets */
                                        /* for a specified interval. */
    MUINT64 uiDropPackets;              /* The total number of dropped by the SRT sender/receiver DATA packets for a */
                                        /* specified interval. */
    MUINT64 uiMbpsRate;                 /* Sending/receiving rate in Mbps for a specified interval. */
    MUINT64 uiSndDurationUs;            /* Same as LNetStreamerSrtStatsAccumulated, but measured on a specified interval. */
    MUINT64 uiRcvUndecryptPackets;      /* Same as LNetStreamerSrtStatsAccumulated, but measured on a specified interval. */
    MUINT64 uiFilterExtraPackets;       /* Same as LNetStreamerSrtStatsAccumulated, but measured on a specified interval. */
    MUINT64 uiRcvFilterSupplyPackets;   /* Same as LNetStreamerSrtStatsAccumulated, but measured on a specified interval. */
    MUINT64 uiRcvFilterLossPackets;     /* Same as LNetStreamerSrtStatsAccumulated, but measured on a specified interval. */
    MUINT64 uiPktReorderDistance;       /* The distance in sequence numbers between the two original (not retransmitted) */
                                        /* packets, that were received out of order. */
    MUINT64 uiRcvBelatedPackets;        /* The number of packets received but IGNORED due to having arrived too late. Makes */
                                        /* sense only if TSBPD and TLPKTDROP are enabled. */
    MUINT64 uiByteSndRcv;               /* The number of packet send/received in bytes on a specific interval. */
    MUINT64 uiByteRcvLoss;              /* The number of packet lost expressed in bytes on a specific interval. */
    MUINT64 uiByteRetrans;              /* The number of packet retransmitted expressed in bytes on a specific interval. */
} LNetStreamerSrtStatsInterval;

/************************************************************************************************************\

Structure:      LNetStreamerSrtStatsInstantaneous

Description:    Instantaneous srt statistics structure.

\************************************************************************************************************/
typedef struct tagLNetStreamerSrtStatsInstantaneous
{
    MUINT64 uiPktSndPeriodUs;           /* Current minimum time interval between which consecutive packets are sent, */
                                        /* in microseconds. Sender only. */
    MUINT64 uiPktFlowWindow;            /* The maximum number of packets that can be "in flight". Sender only. */
                                        /* See also uiPktFlightSize. */
    MUINT64 uiPktCongestionWindow;      /* Congestion window size, in number of packets. Sender only. */
    MUINT64 uiPktFlightSize;            /* The number of packets in flight. Sender only. pktFlightSize <= pktFlowWindow */
                                        /* and pktFlightSize <= pktCongestionWindow. */
    MUINT64 uiRttms;                    /* Smoothed round-trip time (SRTT), an exponentially-weighted moving average (EWMA) */
                                        /* of an endpoint's RTT samples, in milliseconds. */
    MUINT64 uiMbpsBandwidth;            /* Estimated bandwidth of the network link, in Mbps. Sender only. */
    MUINT64 uiAvailBufInBytes;          /* The available space in the sender's/receiver's buffer, in bytes. */
    MUINT64 uiMbpsMaxBandwidth;         /* Transmission bandwidth limit, in Mbps. Sender only. Usually this is the setting */
                                        /* from the SRTO_MAXBW option, which may include the value 0 (unlimited). */
    MUINT64 uiMaxSegmentSizeInBytes;    /* Maximum Segment Size (MSS), in bytes. Same as the value from the SRTO_MSS socket */
                                        /* option. Should not exceed the size of the maximum transmission unit (MTU), in bytes. */
    MUINT64 uiPktBuffer;                /* For sender : The number of packets that are already scheduled for sending or even */
                                        /* possibly sent, but not yet acknowledged. For receiver : The number of acknowledged packets. */
    MUINT64 uiPktBufferMs;              /* For sender : The timespan (msec) of packets (unacknowledged packets). */
                                        /* For receiver : The timespan (msec) of acknowledged packets. */
    MUINT64 uiTsbPdSendDelayMs;         /* Timestamp-based Packet Delivery Delay value of the peer. If SRTO_TSBPDMODE is on */
                                        /* (default for live mode), it returns the value of SRTO_PEERLATENCY, otherwise 0. */
    MUINT64 uiTsbPdRcvDelayMs;          /* Timestamp-based Packet Delivery Delay value of the receiver. If SRTO_TSBPDMODE */
                                        /* is on (default for live mode), it returns the value of SRTO_RCVLATENCY, otherwise 0. */
    MUINT64 uiPktReorderTolerance;      /* Instant value of the packet reorder tolerance. Receiver side. */
                                        /* Refer to pktReorderDistance. */
    MUINT64 uiPktRcvAvgBelatedTime;     /* Accumulated difference between the current time and the time-to-play of a packet */
                                        /* that is received late. */
} LNetStreamerSrtStatsInstantaneous;

/************************************************************************************************************\

Structure:      LNetStreamer_SrtStatistics

Description:    Full srt statistics structure.

\************************************************************************************************************/
typedef struct tagLNetStreamer_SrtStatistics
{
    MBOOL bIsSender;                                    /* Determine if the stats if for a receiver or a sender. */
    LNetStreamerSrtStatsAccumulated oAccumulated;       /* Accumulated part of the SRT statistics. */
    LNetStreamerSrtStatsInterval oInterval;             /* Interval part of the SRT statistics. */
    LNetStreamerSrtStatsInstantaneous oInstantaneous;   /* Instantaneous part of the SRT statistics. */
} LNetStreamer_SrtStatistics;

typedef struct tagLNetStreamer_SrtStatisticsEvent
{
    LNetStreamer_EventType      eEventType;             /* Set this member to LNetStreamer_EventType_PACKET_LOSS. */
    LNetStreamer_MediaType      eMediaType;             /* Report packet loss for this media type. */
    LNetStreamer_SrtStatistics  oSrtStats;              /* Report SRT statistics */
} LNetStreamer_SrtStatisticsEvent;

typedef struct tagLNetStreamer_PacketLossEvent
{
    LNetStreamer_EventType      eEventType;             /* Set this member to LNetStreamer_EventType_PACKET_LOSS. */
    LNetStreamer_MediaType      eMediaType;             /* Report packet loss for this media type. */
    MUINT32                     uiLossPackets;          /* Number of packets lost since last event. */
    MUINT32                     uiExpectedPackets;      /* Number of packets expected since last event. */
    MUINT32                     uiRetransmittedPackets; /* SRT: Number of packets retransmitted since last event. */
    MUINT32                     uiRttUsec;              /* SRT: Detected round-trip-time (RTT) in usec. */
} LNetStreamer_PacketLossEvent;

typedef struct tagLNetStreamer_SrtStreamIDEvent
{
    LNetStreamer_EventType      eEventType;             /* Set this member to LNetStreamer_EventType_STREAMID. */
    MUINT32                     uiLength;               /* Report StreamID length. */
    char*                       pacStreamID;
} LNetStreamer_SrtStreamIDEvent;

typedef LStatus (LNetStreamer_Event)(
    void*                                       pvEventContext,
    LNetStreamer_Handle                         hNetStreamer,
    LNetStreamer_EventType*                     peEventType);


/************************************************************************************************************\

Enum:           LNetStreamer_ConfigType

Description:    Type of network stream configuration structure.

Comments:

\************************************************************************************************************/
typedef enum
{
    LNetStreamer_ConfigType_STANDARD            = 1         /* Standard network stream configuration */
                                                            /* structure.  Set eConfigType to this value for */
                                                            /* LNetStreamer_Config structure. */
} LNetStreamer_ConfigType;


/************************************************************************************************************\

Structure:      LNetStreamer_Config

Description:    Network stream configuration structure.

Comments:       When a video or audio header is optionally set to NULL, then the header is detected from the
                buffers.  The session is not started until all required headers are found.

\************************************************************************************************************/
typedef struct tagLNetStreamer_Config
{
    LNetStreamer_ConfigType     eConfigType;            /* Type of network stream configuration structure.  Set */
                                                        /* this member to LNetStreamer_ConfigType_STANDARD. */
    LNetStreamer_Service        eService;               /* Service of network stream. */
    LNetStreamer_Protocol       eProtocol;              /* Protocol of network stream. */
    MBOOL8                      bEnableVideo;           /* Enable video media. */
    MBOOL8                      bEnableAudio;           /* Enable audio media. */
    MBOOL8                      bEnableMetadata;        /* Enable meta data media. */
    LDevice_Handle              hDevice;                /* Device handle used to create buffers. */
    MUINT32                     uiVideoBufferCount;     /* Number of video buffers to create */
    MUINT32                     uiVideoBufferLength;    /* Length in bytes of each video buffer to create. */
    MUINT32                     uiAudioBufferCount;     /* Number of audio buffers to create. */
    MUINT32                     uiAudioBufferLength;    /* Length in bytes of each audio buffer to create. */
    MUINT32                     uiMetadataBufferCount;  /* Number of meta data buffers to create. */
    MUINT32                     uiMetadataBufferLength; /* Length in bytes of each meta data buffer to create. */
    MUINT32                     uiConfigFlags;          /* Specify members which are present in network stream */
                                                        /* configuration structure. */
    MUINT32                     uiVideoPort;            /* Port to use for video media or for multiplexed video */
                                                        /* and audio media. */
    MUINT32                     uiAudioPort;            /* Port to use for audio media */
    MUINT32                     uiMetadataPort;         /* Port to use for meta data media */
    MCHAR8*                     pacLocation;            /* Pointer to string representing network location of RTSP */
                                                        /* server to connect to. */
    MUINT32                     uiLocationLength;       /* Length of string pointed by pacLocation member. */
    MCHAR8*                     pacDestination;         /* Pointer to string specifying RTP destination address. */
    MUINT32                     uiDestinationLength;    /* Length of string pointed by pacDestination member. */
    MUINT8*                     paucVideoHeaderSps;     /* Pointer to SPS video header. */
    MUINT32                     uiVideoHeaderSpsLength; /* Length of SPS video header. */
    MUINT8*                     paucVideoHeaderPps;     /* Pointer to PPS video header. */
    MUINT32                     uiVideoHeaderPpsLength; /* Length of PPS video header. */
    MUINT8*                     paucAudioHeader;        /* Pointer to audio header. */
    MUINT32                     uiAudioHeaderLength;    /* Length of audio header. */
    MCHAR8*                     pacSdp;                 /* Pointer to string specifying SDP. */
    MUINT32                     uiSdpLength;            /* Length of string pointed by pacSdp member. */
    MCHAR8*                     pacNetInterface;        /* Pointer to string specifying network interface name. */
    MUINT32                     uiNetInterfaceLength;   /* Length of string pointed by pacNetInterface member. */
    MUINT32                     uiRtspPort;             /* Port to use for RTSP server. */
    MCHAR8*                     pacRtspFolder;          /* Pointer to string specifying RTSP folder name. */
    MUINT32                     uiRtspFolderLength;     /* Length of string pointed by pacRtspFolder member. */
    LNetStreamer_Event*         pfnEventCallback;       /* Pointer to callback for application event handler */
    void*                       pvEventContext;         /* Pointer to context for application event handler */
    MBOOL8                      bEnableRtspServer;      /* Enable RTSP server service */
    MBOOL8                      bEnableRtmpServer;      /* Enable RTMP server service */
    MUINT32                     uiRtcpFrequency;        /* Number of milliseconds between RTCP reports. */
    MUINT8*                     paucMasterKey;          /* Pointer to SRTP master key */
    MUINT32                     uiMasterKeyLength;      /* Length of SRTP master key */
    MUINT8*                     paucMasterSalt;         /* Pointer to SRTP master salt */
    MUINT32                     uiMasterSaltLength;     /* Length of SRTP master salt */
    MUINT32                     uiTtl;                  /* Specify TTL configuration option. */
    MCHAR8*                     pacRtmpApp;             /* Pointer to string representing RTMP server */
                                                        /* application name. */
    MUINT32                     uiRtmpAppLength;        /* Length of RTMP server application name. */
    MUINT32                     uiMaxBitrateKbps;       /* Maximum bitrate in Kbps. */
    MCHAR8*                     pacUsername;            /* Pointer to RTSP authentication user name. */
    MUINT32                     uiUsernameLength;       /* Length of string pointed by pacUsername member. */
    MCHAR8*                     pacPassword;            /* Pointer to RTSP authentication password. */
    MUINT32                     uiPasswordLength;       /* Length of string pointed by pacPassword member. */
    MUINT32                     uiMtu;                  /* Maximum size in bytes of UDP packets. */
    MUINT32                     uiServerVideoPort;      /* Port to use for video media or for multiplexed video */
                                                        /* and audio media. */
    MUINT32                     uiServerAudioPort;      /* Port to use for audio media */
    MUINT32                     uiConfigFlags2;         /* Extension of uiConfigFlags member. */
    MUINT32                     uiSrtLatencyMsec;       /* Specify latency in milliseconds of SRT protocol. */
    MUINT32                     uiSrtKeyLength;         /* Specify length in bytes of SRT encryption key. */
    MINT32                      iNiceIncrement;         /* Specify increment of thread niceness. */
    MUINT32                     uiSSRC;                 /* Specify SSRC associated with RTP stream */
    MUINT32                     uiRtmpTimeout;          /* Timeout in seconds for RTMP protocol. */
    MCHAR8*                     pacHlsFolder;           /* Pointer to string for HLS folder path */
    MUINT32                     uiHlsFolderLength;      /* Length of pacHlsFolder */
    MUINT32                     uiHlsDuration;          /* HLS segment duration in milli-seconds. */
    MUINT32                     uiHlsCount;             /* HLS segment count. */
    MCHAR8*                     pacHlsMaster;           /* Pointer to string for HLS master name */
    MUINT32                     uiHlsMasterLength;      /* Length of pacHlsMaster. */
    MUINT32                     uiVideoFrameRateNum;    /* Numerator of video frame rate in Hz */
    MUINT32                     uiVideoFrameRateDen;    /* Denominator of video frame rate in Hz */
    MUINT32                     uiVideoTrackCount;      /* Number of video tracks */
    MUINT32                     uiAudioTrackCount;      /* Number of audio tracks */
    LNetStreamer_SrtMode        eSrtMode;               /* SRT connection mode */
    MCHAR8*                     pacSource;              /* Pointer to source address for IGMP v3 SSM */
    MUINT32                     uiSourceLength;         /* Length of buffer pointed by pacSource member. */
    MUINT32                     uiTos;                  /* Specify IP layer type of service value. */
    MUINT32                     auiAudioTrackPid[LNETSTREAMER_MAX_NUMBER_OF_AUDIO_TRACK];   /* Array containing the audio tracks PID */
    MUINT32                     uiMetadataTrackCount;   /* Number of metadata tracks */
    MUINT32                     uiMetadataLength;       /* Length of the first dectected packet */
    MCHAR8*                     pacStreamID;            /* Pointer to SRT streamID. */
    MUINT32                     uiStreamIDLength;       /* Length of string pointed by pacStreamID member. */
} LNetStreamer_Config;

/************************************************************************************************************\

Enum:           LNetStreamer_RtspMethod

Description:    RTSP method

Comments:

\************************************************************************************************************/
typedef enum tagLNetStreamer_RtspMethod
{
    LNetStreamer_RtspMethod_NONE           = 0,
    LNetStreamer_RtspMethod_OPTIONS        = 1,
    LNetStreamer_RtspMethod_DESCRIBE       = 2,
    LNetStreamer_RtspMethod_SETUP          = 3,
    LNetStreamer_RtspMethod_PLAY           = 4,
    LNetStreamer_RtspMethod_PAUSE          = 5,
    LNetStreamer_RtspMethod_TEARDOWN       = 6,
    LNetStreamer_RtspMethod_GET_PARAMETER  = 7,
    LNetStreamer_RtspMethod_SET_PARAMETER  = 8,
    LNetStreamer_RtspMethod_ANNOUNCE       = 9,
    LNetStreamer_RtspMethod_RECORD         = 10
} LNetStreamer_RtspMethod;

/* -------------------------------------------------------------------------------------------------------- */
/*                           G L O B A L   V A R I A B L E   R E F E R E N C E S */
/* -------------------------------------------------------------------------------------------------------- */

/* -------------------------------------------------------------------------------------------------------- */
/*                      I N L I N E S   A N D   T E M P L A T E   D E F I N I T I O N S */
/* -------------------------------------------------------------------------------------------------------- */


#if defined(__cplusplus)
extern "C" {
#endif /* #if defined(__cplusplus) */


/************************************************************************************************************\

Function:       LNetStreamer_Create

Description:    Create a network stream handle.

                Create a network stream handle which is used to transfer media.  When the transfer of media is
                completed the application may release the network stream handle by calling LNetStreamer_Destroy.

Parameters:     peConfigType [in]               Network stream creation options.  Pointer to structure that
                                                has LNetStreamer_ConfigType as first member.
                phNetStreamer [out]             Pointer to returned network stream handle.

Return Value:   LStatus_OK                      Function completed successfully.
                LStatus_INVALID_PARAM           Function failed due to invalid parameters.
                LStatus_OUT_OF_MEMORY           Insufficient memory.
                LStatus_FAIL                    Function failed.
                LStatus_OK_MISSING_FIRST_SPS    Function completed successfully but the network stream is not
                                                completely configured.  Send some media to complete the
                                                configuration.
                LStatus_ACCESS_DENIED           Function failed after authentication attempt.

Comments:       On successful creation, the transfer of buffers begins.  The application must be ready to
                start processing buffers on return by calling LNetStreamer_GetMedia, LNetStreamer_SendMedia or
                LNetStreamer_ReceiveMedia.

                For server streams, when not all headers are supplied via the stream creation options
                structure, the stream handle is created but the transfer of buffers does not begin until all
                headers are found via the LNetStreamer_GetMedia and LNetStreamer_ReleaseMedia functions.

\************************************************************************************************************/
LAPI LStatus LNetStreamer_Create(
    LNetStreamer_ConfigType*                    peConfigType,
    LNetStreamer_Handle*                        phNetStreamer);

/************************************************************************************************************\

Function:       LNetStreamer_Destroy

Description:    Destroy a network stream handle.

                Free all resources of network stream.

Parameters:     hNetStreamer [in]               Handle of network stream

Return Value:   LStatus_OK                      Function completed successfully.
                LStatus_INVALID_PARAM           Function failed due to invalid parameters.
                LStatus_FAIL                    Function failed.

Comments:       Network stream handle is invalid on return.  Application must clear all references to the
                network stream handle prior to calling this function.

                Using the network handle after destruction in other LNetStreamer functions will produce
                undefined behaviour.

\************************************************************************************************************/
LAPI LStatus LNetStreamer_Destroy(
    LNetStreamer_Handle                         hNetStreamer);

/************************************************************************************************************\

Function:       LNetStreamer_SendRtspClientRequest

Description:    Send an RTSP request via the RTSP player client.
                If successfully sent retrieve the CSeq header value for later use.

Parameters:     IN  hNetStreamer                Handle of network stream to send message from.
                IN  pacRequestText              Request text
                IN  uiRequestLength             Length of the request text
                IN  eRtspMethod                 RTSP Method type of the request
                OUT puiCSequenceNumber          RTSP protocol header value used to map responses to requests

Return Value:   LStatus_OK                      Function completed successfully.
                LStatus_UNSUPPORTED             Sending RTSP requests is not supported by this network stream
                LStatus_FAIL                    Function failed.

Comments:

\************************************************************************************************************/
LAPI LStatus LNetStreamer_SendRtspClientRequest(
    LNetStreamer_Handle                         hNetStreamer,
    MCHAR8*                                     pacRequestText,
    MUINT32                                     uiRequestLength,
    LNetStreamer_RtspMethod                     eRtspMethod,
    MUINT32*                                    puiCSequenceNumber);

/************************************************************************************************************\

Function:       LNetStreamer_GetMedia

Description:    Get media information structure.

                Get media information for given media type.  For client streams, the data length and timestamp
                members will be returned.  For server streams, the data length and timestamp members are not
                returned.  If no buffer is available then this function will wait for a buffer to become
                available.  Once finished using the returned media information, release it by calling
                LNetStreamer_ReleaseMedia.

Parameters:     hNetStreamer [in]               Handle of network stream
                peMediaType [out]               Pointer to media information.  Pointer to a structure that has
                                                LNetStreamer_MediaType as its first member.

Return Value:   LStatus_OK                      Function completed successfully.
                LStatus_INVALID_PARAM           Function failed due to invalid parameters.
                LStatus_FAIL                    Function failed.
                LStatus_OUT_OF_RESOURCES        Unable to find any available buffers.  Call LNetStreamer_-
                                                ReleaseMedia to free some resources and try again.  The
                                                maximum number of buffers available may be configured using
                                                the uiBufferCount member of the stream creation options
                                                structure.
                LStatus_OUT_OF_MEMORY           Insufficient memory.
                LStatus_OK_MISSING_FIRST_SPS    Function completed successfully but the network stream is not
                                                completely configured.  Continue sending some buffers to
                                                complete the configuration.
                LStatus_TIMEOUT                 Function failed due to timeout.

Comments:       When the return value is not equal to LStatus_OK, the application is expected to retry by
                calling LNetStreamer_GetMedia.  The retry mechanism is necessary to avoid blocking the
                application for a long period of time.

                For client streams, initialize eMediaType member of media information structure before calling
                LNetStreamer_GetMedia function.  If LNetStreamer_GetMedia function returns LStatus_OK, do not
                modify any of the media information structure members until LNetStreamer_ReleaseMedia is
                called.

                For server streams, initialize eMediaType member of media information structure before calling
                LNetStreamer_GetMedia function.  If LNetStreamer_GetMedia function returns LStatus_OK,
                modify only the data length and timestamp members until LNetStreamer_ReleaseMedia is called,
                all other members must not be modified.

\************************************************************************************************************/
LAPI LStatus LNetStreamer_GetMedia(
    LNetStreamer_Handle                         hNetStreamer,
    LNetStreamer_MediaType*                     peMediaType);


/************************************************************************************************************\

Function:       LNetStreamer_ReleaseMedia

Description:    Release media information structure.

                Indicate that the application is finished using a media information structure.  For a server
                network stream, the media is sent over the network using the information supplied by the media
                information structure.

Parameters:     hNetStreamer [in]               Handle of network stream
                peMediaType [in]                Pointer to media information.  Pointer to a structure that has
                                                LNetStreamer_MediaType as its first member.

Return Value:   LStatus_OK                      Function completed successfully.
                LStatus_INVALID_PARAM           Function failed due to invalid parameters.
                LStatus_FAIL                    Function failed.
                LStatus_OK_MISSING_FIRST_SPS    Function completed successfully but the network stream is not
                                                completely configured.  Continue sending some media to
                                                complete the configuration.

Comments:       Make sure to copy all members of media information structure when making copies.

\************************************************************************************************************/
LAPI LStatus LNetStreamer_ReleaseMedia(
    LNetStreamer_Handle                         hNetStreamer,
    LNetStreamer_MediaType*                     peMediaType);


/************************************************************************************************************\

Function:       LNetStreamer_GetConfigLength

Description:    Get length in bytes of stream configuration structure.

                Get required length for network stream configuration structure.

Parameters:     hNetStreamer [in]               Handle of network stream
                eConfigType [in]                Type of network stream configuration structure
                puiLength [out]                 Pointer to returned network stream configuration length.

Return Value:   LStatus_OK                      Function completed successfully.
                LStatus_INVALID_PARAM           Function failed due to invalid parameters.
                LStatus_FAIL                    Function failed.

Comments:       This function may fail when the network stream is not completely configured.  If that is the
                case, continue sending media and try again later.

\************************************************************************************************************/
LAPI LStatus LNetStreamer_GetConfigLength(
    LNetStreamer_Handle                         hNetStreamer,
    LNetStreamer_ConfigType                     eConfigType,
    MUINT32*                                    puiConfigLength);


/************************************************************************************************************\

Function:       LNetStreamer_GetConfig

Description:    Get network stream configuration structure.

                Fill given buffer with network stream configuration information.  The returned pointer
                provides details about the buffer’s contents.

Parameters:     hNetStreamer [in]               Handle of network stream
                peConfigType [out]              Pointer to buffer large enough to contain network stream
                                                configuration information.  The first member must be set to
                                                appropriate value of LNetStreamer_ConfigType enumeration.
                uiLength [in]                   Length of buffer pointed by peConfigType parameter.

Return Value:   LStatus_OK                      Function completed successfully.
                LStatus_INVALID_PARAM           Function failed due to invalid parameters.
                LStatus_FAIL                    Function failed.

Comments:       This function may fail when the network stream is not completely configured.  If that is the
                case, continue sending media and try again later.

\************************************************************************************************************/
LAPI LStatus LNetStreamer_GetConfig(
    LNetStreamer_Handle                         hNetStreamer,
    LNetStreamer_ConfigType*                    peConfigType,
    MUINT32                                     uiConfigLength);


/************************************************************************************************************\

Function:       LNetStreamer_SendMedia

Description:    Send media information structure.

                Send media information to network.  Application provides memory for data and specifies data
                length and timestamps.

Parameters:     hNetStreamer [in]               Handle of network stream
                peMediaType [in]                Pointer to media information.  Pointer to a structure that has
                                                LNetStreamer_MediaType as its first member.

Return Value:   LStatus_OK                      Function completed successfully.
                LStatus_INVALID_PARAM           Function failed due to invalid parameters.
                LStatus_FAIL                    Function failed.
                LStatus_OUT_OF_RESOURCES        Unable to find any available buffers.  Call LNetStreamer_-
                                                ReleaseMedia to free some resources and try again.  The
                                                maximum number of buffers available may be configured using
                                                the uiBufferCount member of the network stream creation
                                                options structure.
                LStatus_OUT_OF_MEMORY           Insufficient memory.
                LStatus_OK_MISSING_FIRST_SPS    Function completed successfully but the network stream is not
                                                completely configured.  Continue sending some buffers to
                                                complete the configuration.

Comments:       This method is only for server network streams, as an alternative to LNetStreamer_GetMedia and
                LNetStreamer_ReleaseMedia.

                This method is blocking until buffer has finished being accessed.

\************************************************************************************************************/
LAPI LStatus LNetStreamer_SendMedia(
    LNetStreamer_Handle                         hNetStreamer,
    LNetStreamer_MediaType*                     peMediaType);


/************************************************************************************************************\

Function:       LNetStreamer_ReceiveMedia

Description:    Receive media information structure.

                Receive media information from network.  Application provides memory for buffer and method
                returns buffer information including actual data length and timestamps.

Parameters:     hNetStreamer [in]               Handle of network stream
                peMediaType [in]                Pointer to media information.  Pointer to a structure that has
                                                LNetStreamer_MediaType as its first member.

Return Value:   LStatus_OK                      Function completed successfully.
                LStatus_INVALID_PARAM           Function failed due to invalid parameters.
                LStatus_FAIL                    Function failed.
                LStatus_OUT_OF_RESOURCES        Unable to find any available buffers.  Call LNetStreamer_-
                                                ReleaseMedia to free some resources and try again.  The
                                                maximum number of buffers available may be configured using
                                                the uiBufferCount member of the network stream creation
                                                options structure.
                LStatus_OUT_OF_MEMORY           Insufficient memory.
                LStatus_TIMEOUT                 Function failed due to a timeout.

Comments:       This method is only for client network streams, as an alternative to LNetStreamer_GetMedia and
                LNetStreamer_ReleaseMedia.

                The data that is referenced by the media information structure will only be accessed from
                within this function.  The contents of the data may be modified even when the function returns
                an error.

                When the return value is equal to LStatus_TIMEOUT, the application is expected to retry by
                calling LNetStreamer_ReceiveMedia.  The retry mechanism is necessary to avoid blocking the
                application for a long period of time.

\************************************************************************************************************/
LAPI LStatus LNetStreamer_ReceiveMedia(
    LNetStreamer_Handle                         hNetStreamer,
    LNetStreamer_MediaType*                     peMediaType);


#if defined(__cplusplus)
} /* extern "C" */
#endif /* #if defined(__cplusplus) */


#endif /* #ifndef INC_LNETSTREAMER_H */

